// DH
// Benne de Weger - TU/e - February 2021

// FAECTOR Workshop Cryptographic Programming
// Assignment 14 - application
//    // constructor
//    public DH() throws Exception
//    public DH(String id) throws Exception
//    // generate system parameters
//    public void generateDHSystemParameters(int b)
//    // generate key pair
//    public void generateDHKeyPair() throws Exception
//    // write DH system parameters to file fn
//    public void writeDHSystemParameters(String fn) throws Exception
//    // write DH public key to file fn
//    public void writeDHPublicKey(String fn) throws Exception
//    // write DH private key to file fn
//    public void writeDHPrivateKey(String fn) throws Exception
//    // write encrypted DH private key to file fn
//    public void secureWriteDHPrivateKey(String password, String fn) throws Exception
//    // read DH system parameters from file fn
//    public void readDHSystemParameters(String fn) throws Exception
//    // read DH public key from file fn
//    public void readOwnDHPublicKey(String fn) throws Exception
//    // read DH public key from file fn
//    public void readPartnerDHPublicKey(String fn, String id_partner) throws Exception
//    // read DH private key from file fn
//    public void readDHPrivateKey(String fn) throws Exception
//    // read encrypted DH private key from file fn
//    public void secureReadDHPrivateKey(String password, String fn) throws Exception
//    // derive shared secret
//    public void deriveSharedSecret() throws Exception

import net.deweger.crypto.*;
import java.math.BigInteger;
import java.util.List;
import java.util.ArrayList;
import java.nio.file.*;
import java.nio.charset.StandardCharsets;

public class DH
{
    private String id;
    private BigInteger p = BigInteger.ONE;
    private BigInteger g = BigInteger.ONE;
    private BigInteger x_own = BigInteger.ONE;
    private BigInteger y_own = BigInteger.ONE;
    private BigInteger y_partner = BigInteger.ONE;
    public byte[] s;

    Util util;
    SHA256 sha256;
    PBE pbe;

    // constructor
    public DH() throws Exception
    {
        util = new Util();
        sha256 = new SHA256();
        pbe = new PBE();
    }

    // constructor
    public DH(String id) throws Exception
    {
        this.id = id;
        util = new Util();
        sha256 = new SHA256();
        pbe = new PBE();
    }

    // generate system parameters
    public void generateDHSystemParameters(int b)
    {
        // prime modulus p and generator g
        // p is a Sophie Germain prime
        p = SophieGermain.generateSophieGermainPrime(b);
        g = new BigInteger(b, util.secureRandom).mod(p);
    }

    // generate key pair
    public void generateDHKeyPair() throws Exception
    {
        if (p.equals(BigInteger.ONE))
            throw new Exception("DH.generateDHKeyPair: system parameters not set");
        x_own = new BigInteger(p.bitLength(), util.secureRandom).mod(p.subtract(BigInteger.ONE));
        y_own = g.modPow(x_own, p);
    }

    // write DH system parameters to file fn
    public void writeDHSystemParameters(String fn) throws Exception
    {
        ArrayList<String[]> param = new ArrayList<String[]>();

        // build the tag-value structure
        // prime modulus
        String[] prime = new String[2];
        prime[0] = "[p] (prime modulus)";
        prime[1] = Util.bigIntegerToHex64(p);
        param.add(prime);

        // generator
        String[] generator = new String[2];
        generator[0] = "[g] (generator)";
        generator[1] = Util.bigIntegerToHex64(g);
        param.add(generator);

        // write to file
        List<String> lines = new ArrayList<String>();
        for (String[] item: param)
            lines.add(item[0] + "\n" + item[1]);
        Files.write(Paths.get(fn), lines, StandardCharsets.UTF_8, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
    }

    // write DH public key to file fn
    public void writeDHPublicKey(String fn) throws Exception
    {
        ArrayList<String[]> key = new ArrayList<String[]>();

        // build the tag-value structure
        // identity
        String[] identity = new String[2];
        identity[0] = "[id] (identity)";
        identity[1] = id;
        key.add(identity);
        // public key
        String[] publicKey = new String[2];
        publicKey[0] = "[y] (public key)";
        publicKey[1] = Util.bigIntegerToHex64(y_own);
        key.add(publicKey);

        // write to file
        List<String> lines = new ArrayList<String>();
        for (String[] item: key)
            lines.add(item[0] + "\n" + item[1]);
        Files.write(Paths.get(fn), lines, StandardCharsets.UTF_8, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
    }

    // prepare DH private key for writing
    private List<String> prepareDHPrivateKey() throws Exception
    {
        ArrayList<String[]> key = new ArrayList<String[]>();

        // build the tag-value structure
        // identity
        String[] identity = new String[2];
        identity[0] = "[id] (identity)";
        identity[1] = id;
        key.add(identity);
        // private key
        String[] privateKey = new String[2];
        privateKey[0] = "[x] (private key)";
        privateKey[1] = Util.bigIntegerToHex64(x_own);
        key.add(privateKey);

        // prepare for write to file
        List<String> lines = new ArrayList<String>();
        for (String[] item: key)
            lines.add(item[0] + "\n" + item[1]);
        return lines;
    }

    // write DH private key to file fn
    public void writeDHPrivateKey(String fn) throws Exception
    {
        // prepare key
        List<String> lines = prepareDHPrivateKey();

        // write key
        Files.write(Paths.get(fn), lines, StandardCharsets.UTF_8, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
    }

    // write encrypted DH private key to file fn
    public void secureWriteDHPrivateKey(String password, String fn) throws Exception
    {
        // prepare key
        List<String> lines = prepareDHPrivateKey();

        // encrypt key under password
        String serialized = ""; // lines separator is "@#"
        for (String line: lines)
            serialized += line + "@#";
        pbe.setPassword(password);
        pbe.setSalt("DHPrivateKey".getBytes());
        pbe.setIterationCount(1000);
        String[] encrypted = Util.bytesToHex64(pbe.encrypt(serialized.getBytes())).split("\n");
        ArrayList<String> encryptedLines = new ArrayList<String>();
        for (int i = 0; i < encrypted.length; i++)
            encryptedLines.add(encrypted[i]);

        // write encrypted key
        Files.write(Paths.get(fn), encryptedLines, StandardCharsets.UTF_8, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
    }

    // read DH system parameters from file fn
    public void readDHSystemParameters(String fn) throws Exception
    {
        ArrayList<String[]> key = new ArrayList<String[]>();

        // read file
        List<String> lines = Files.readAllLines(Paths.get(fn));

        // parse into tag-value structure
        String[] tagValue = new String[2];
        boolean first = true;
        for(String line: lines)
        {
            if (line.startsWith("["))
            {
                if (first)
                    first = false;
                else
                    key.add(tagValue);
                tagValue = new String[2];
                tagValue[0] = line;
                tagValue[1] = "";
            }
            else
                tagValue[1] += line.trim();
        }
        key.add(tagValue);

        // set variables
        for (String[] s: key)
        {
            if (s[0].startsWith("[p]"))
                p = new BigInteger(s[1], 16);
            if (s[0].startsWith("[g]"))
                g = new BigInteger(s[1], 16);
        }
    }

    // read DH public key from file fn
    public void readOwnDHPublicKey(String fn) throws Exception
    {
        String id2 = "";

        ArrayList<String[]> key = new ArrayList<String[]>();

        // read file
        List<String> lines = Files.readAllLines(Paths.get(fn));

        // parse into tag-value structure
        String[] tagValue = new String[2];
        boolean first = true;
        for(String line: lines)
        {
            if (line.startsWith("["))
            {
                if (first)
                    first = false;
                else
                    key.add(tagValue);
                tagValue = new String[2];
                tagValue[0] = line;
                tagValue[1] = "";
            }
            else
                tagValue[1] += line.trim();
        }
        key.add(tagValue);

        // set variables
        for (String[] s: key)
        {
            if (s[0].startsWith("[id]"))
            {
                id2 = s[1];
                if (!id.equals(id2))
                    throw new Exception("DH.readOwnDHPublicKey: identities do not match");
            }
            if (s[0].startsWith("[y]"))
                y_own = new BigInteger(s[1], 16);
        }
    }

    // read DH public key from file fn
    public void readPartnerDHPublicKey(String fn, String id_partner) throws Exception
    {
        String id2 = "";

        ArrayList<String[]> key = new ArrayList<String[]>();

        // read file
        List<String> lines = Files.readAllLines(Paths.get(fn));

        // parse into tag-value structure
        String[] tagValue = new String[2];
        boolean first = true;
        for(String line: lines)
        {
            if (line.startsWith("["))
            {
                if (first)
                    first = false;
                else
                    key.add(tagValue);
                tagValue = new String[2];
                tagValue[0] = line;
                tagValue[1] = "";
            }
            else
                tagValue[1] += line.trim();
        }
        key.add(tagValue);

        // set variables
        for (String[] s: key)
        {
            if (s[0].startsWith("[id]"))
            {
                id2 = s[1];
                if (!id_partner.equals(id2))
                    throw new Exception("DH.readPartnerDHPublicKey: identities do not match");
            }
            if (s[0].startsWith("[y]"))
                y_partner = new BigInteger(s[1], 16);
        }
    }

    // process DH private key after reading
    private void processDHPrivateKey(List<String> lines) throws Exception
    {
        String id2 = "";

        ArrayList<String[]> key = new ArrayList<String[]>();

        // parse into tag-value structure
        String[] tagValue = new String[2];
        boolean first = true;
        for(String line: lines)
        {
            if (line.startsWith("["))
            {
                if (first)
                    first = false;
                else
                    key.add(tagValue);
                tagValue = new String[2];
                tagValue[0] = line;
                tagValue[1] = "";
            }
            else
                tagValue[1] += line.trim();
        }
        key.add(tagValue);

        // set variables
        for (String[] s: key)
        {
            if (s[0].startsWith("[id]"))
            {
                id2 = s[1];
                if (!id.equals(id2))
                    throw new Exception("DH.processDHPrivateKey: identities do not match");
            }
            if (s[0].startsWith("[x]"))
                x_own = new BigInteger(s[1], 16);
        }
    }

    // read DH private key from file fn
    public void readDHPrivateKey(String fn) throws Exception
    {
        // read key
        List<String> lines = Files.readAllLines(Paths.get(fn));

        // process key
        processDHPrivateKey(lines);
    }

    // read encrypted DH private key from file fn
    public void secureReadDHPrivateKey(String password, String fn) throws Exception
    {
        // read encrypted key
        List<String> encryptedLines = Files.readAllLines(Paths.get(fn));

        // decrypt key under password
        String encrypted = "";
        for (String line: encryptedLines)
            encrypted += line;
        pbe.setPassword(password);
        pbe.setSalt("DHPrivateKey".getBytes());
        pbe.setIterationCount(1000);
        String[] serialized = (new String(pbe.decrypt(Util.hexToBytes(encrypted)), StandardCharsets.UTF_8)).split("@#");
        ArrayList<String> lines = new ArrayList<String>();
        for (int i = 0; i < serialized.length; i++)
        {
            String[] s = serialized[i].split("\n");
            for (int j = 0; j < s.length; j++)
                lines.add(s[j]);
        }

        // process key
        processDHPrivateKey(lines);
    }

    // derive shared secret
    public void deriveSharedSecret() throws Exception
    {
        if (p.equals(BigInteger.ONE))
            throw new Exception("DH.deriveSharedSecret: system parameters not set");
        if (x_own.equals(BigInteger.ONE))
            throw new Exception("DH.deriveSharedSecret: own private key not set");
        if (y_partner.equals(BigInteger.ONE))
            throw new Exception("DH.deriveSharedSecret: partner public key not set");
        s = sha256.hash(y_partner.modPow(x_own, p).toByteArray());
    }
}
