// PBE class - Password Based Encryption
// using AES (128-bit key) in CTR mode with PKCS#7 padding,
// and HMAC-SHA256
// Benne de Weger - TU/e - January 2021

// FAECTOR Workshop Cryptographic Programming
// Assignment 1, 2 - API:
//
// constructor
//     public PBE() throws Exception
// key derivation
//     public byte[] PBKDF2(String pw, byte[] st, int ic, int dL) throws Exception
// set functions
//     public void setPassword(String p)
//     public void setSalt(byte[] s)
//     public void setIterationCount(int i)
//     public void setIv(byte[] i) throws Exception
// get functions
//     public String getPassword()
//     public byte[] getSalt()
//     public int getIterationCount()
//     public byte[] getIv() throws Exception
// encrypt and decrypt
//     public byte[] encrypt(byte[] plaintext) throws Exception
//     public byte[] decrypt(byte[] ciphertext) throws Exception

import net.deweger.crypto.*;

import java.math.BigInteger;

public class PBE
{
    private AES aes;
    private SHA256 sha256;

    private String password = "";
    private byte[] salt = new byte[0];
    private int iterationCount = 0;

    private byte[] iv = new byte[0];

    private int dL = 16;
    private int hL = 32;

    // constructor
    public PBE() throws Exception
    {
        sha256 = new SHA256();
        password = "";         // default password is an empty string
        salt = new byte[0];    // default salt is an empty byte array
        iterationCount = 1000; // default iteration count = 1000
        iv = new byte[16];     // default iv is the all-zero iv

        aes = new AES();
        aes.setMode(AES.MODE_CTR);
        aes.setPadding(AES.PAD_PKCS7);
    }

    // Key derivation, hL = 32 (number of bytes in HMAC)
    public byte[] PBKDF2(String pw, byte[] st, int ic, int dL) throws Exception
    {
        int l = (dL + hL - 1) / hL;
        int r = dL - (l-1) * hL;
        byte dk[] = new byte[dL];
        for (int k = 1; k <= l; k++)
        {
            byte[] st1 = new byte[st.length + 4];
            System.arraycopy(st, 0, st1, 0, st.length);
            st1[st.length + 3] = (byte)k;
            byte[] pwb = pw.getBytes();
            byte[] u = st1;
            byte[] f = new byte[hL];
            for (int i = 0; i < ic; i++)
            {
                u = sha256.hMac(pwb, u);
                for (int j = 0; j < hL; j++)
                    f[j] = (byte)(f[j] ^ u[j]);
            }
            if (k < l)
                System.arraycopy(f, 0, dk, (k-1) * hL, hL);
            else
                System.arraycopy(f, 0, dk, (k-1) * hL, r);
        }
        return dk;
    }

    // set password
    public void setPassword(String p)
    {
        password = p;
    }

    // set salt
    public void setSalt(byte[] s)
    {
        salt = s;
    }

    // set iteration count
    public void setIterationCount(int i)
    {
        iterationCount = i;
    }

    // set iv
    public void setIv(byte[] i) throws Exception
    {
        if (i.length == 16)
            iv = i;
        else throw new Exception("PBE.setIv: illegal iv length: " + i.length);
    }

    // get password
    public String getPassword()
    {
        return password;
    }

    // get salt
    public byte[] getSalt()
    {
        return salt;
    }

    // get iteration count
    public int getIterationCount()
    {
        return iterationCount;
    }

    // get iv
    public byte[] getIv() throws Exception
    {
        if (iv.length != 16)
            throw new Exception("PBE.getIv: illegal iv length: " + iv.length);
        return iv;
    }

    public byte[] encrypt(byte[] plaintext) throws Exception
    {
        aes.setKey(PBKDF2(password, salt, iterationCount, dL));
        if (iv.length == 16)
            aes.setIv(iv);
        else throw new Exception("PBE.encrypt: illegal iv length: " + iv.length);
        return aes.encrypt(plaintext);
    }

    public byte[] decrypt(byte[] ciphertext) throws Exception
    {
        aes.setKey(PBKDF2(password, salt, iterationCount, dL));
        if (iv.length == 16)
            aes.setIv(iv);
        else throw new Exception("PBE.decrypt: illegal iv length: " + iv.length);
        return aes.decrypt(ciphertext);
    }
}
