package net.deweger.crypto;

import java.io.*;

// SHA-256 class.
// Benne de Weger
// version 1.1, January 2021
public class SHA256
{
    // constants
    private static long[] y = {0x428A2F98L, 0x71374491L, 0xB5C0FBCFL, 0xE9B5DBA5L, 0x3956C25BL, 0x59F111F1L, 0x923F82A4L, 0xAB1C5ED5L,
                               0xD807AA98L, 0x12835B01L, 0x243185BEL, 0x550C7DC3L, 0x72BE5D74L, 0x80DEB1FEL, 0x9BDC06A7L, 0xC19BF174L,
                               0xE49B69C1L, 0xEFBE4786L, 0x0FC19DC6L, 0x240CA1CCL, 0x2DE92C6FL, 0x4A7484AAL, 0x5CB0A9DCL, 0x76F988DAL,
                               0x983E5152L, 0xA831C66DL, 0xB00327C8L, 0xBF597FC7L, 0xC6E00BF3L, 0xD5A79147L, 0x06CA6351L, 0x14292967L,
                               0x27B70A85L, 0x2E1B2138L, 0x4D2C6DFCL, 0x53380D13L, 0x650A7354L, 0x766A0ABBL, 0x81C2C92EL, 0x92722C85L,
                               0xA2BFE8A1L, 0xA81A664BL, 0xC24B8B70L, 0xC76C51A3L, 0xD192E819L, 0xD6990624L, 0xF40E3585L, 0x106AA070L,
                               0x19A4C116L, 0x1E376C08L, 0x2748774CL, 0x34B0BCB5L, 0x391C0CB3L, 0x4ED8AA4AL, 0x5B9CCA4FL, 0x682E6FF3L,
                               0x748F82EEL, 0x78A5636FL, 0x84C87814L, 0x8CC70208L, 0x90BEFFFAL, 0xA4506CEBL, 0xBEF9A3F7L, 0xC67178F2L};
    private static long[] iv = {0x6A09E667L, 0xBB67AE85L, 0x3C6EF372L, 0xA54FF53AL, 0x510E527FL, 0x9B05688CL, 0x1F83D9ABL, 0x5BE0CD19L};

    // context
    private long[] state;
    private long[] intState;
    private long   count;
    private byte[] buffer;
    private int    bufferFilled;
    private byte[] padding;
    private byte[] hashValue;
    
    // Convert the state to a byte array.
    private byte[] stateToBytes(long[] st)
    {
        byte[] out = new byte[st.length*4];
        for (int i = 0; i < st.length; i++)
            for (int j = 0; j < 4; j++)
                out[4*(i+1)-j-1] = (byte)((st[i]>>>(8*j))&0xFFL);
        return out;
    }
    
    // f(u,v,w) = (u&v)|((~u)&w)
    private static long f(long u, long v, long w)
    {
        return (u & v) | ((~u) & w);
    }

    // g(u,v,w) = (u&v)|(u&w)|(v&w)
    private static long g(long u, long v, long w)
    {
        return (u & v) | (u & w) | (v & w);
    }

    // Cyclic bitshift of 32 bits.
    private static long shift32(long t, int s)
    {
        return ((t<<s) | (t>>>(32L-s))) & 0xFFFFFFFFL;
    }

    // shift
    private static long shift(long t, int s)
    {
        return shift32(t, s);
    }

    // Sigma
    private long Sigma(int r1, int r2, int r3, long x)
    {
        return shift(x, 32-r1) ^ shift(x, 32-r2) ^ shift(x, 32-r3);
    }

    // sigma
    private long sigma(int r1, int r2, int r3, long x)
    {
        return shift(x, 32-r1) ^ shift(x, 32-r2) ^ (x >>> r3);
    }

    // compression function
    private void compress()
    {
        long[] x = new long[64];
        for(int i = 0; i < 16; i++)
        {
            long[] xx = new long[4];
            for (int j = 0; j < 4; j++)
            {
                xx[j] = (long)buffer[i*4+j];
                if (xx[j] < 0) xx[j] += 256;
            }
            x[i] = xx[3] + (xx[2] << 8) + (xx[1] << 16) + (xx[0] << 24);
        }
        for(int i = 16; i < 64; i++)
            x[i] = (sigma(17, 19, 10, x[i-2]) + x[i-7] + sigma(7, 18, 3, x[i-15]) + x[i-16]) & 0xFFFFFFFFL;
        for (int i = 0; i < 8; i++)
            intState[i] = state[i];
        for(int j = 0; j < 64; j++)
        {
            long t1 = (intState[7] + Sigma(6, 11, 25, intState[4]) + f(intState[4], intState[5], intState[6]) + x[j] + y[j]) & 0xFFFFFFFFL;
            long t2 = (Sigma(2, 13, 22, intState[0]) + g(intState[0], intState[1], intState[2])) & 0xFFFFFFFFL;
            intState[7] = intState[6]; intState[6] = intState[5]; intState[5] = intState[4];
            intState[4] = (intState[3] + t1) & 0xFFFFFFFFL;
            intState[3] = intState[2]; intState[2] = intState[1]; intState[1] = intState[0];
            intState[0] = (t1 + t2) & 0xFFFFFFFFL;
        }
        for (int i = 0; i < 8; i++)
            state[i] = (state[i] + intState[i]) & 0xFFFFFFFFL;
    }

    // Computes hash padding.
    private void pad()
    {
        int r = 56 - bufferFilled;
        if (r <= 0) r += 64;
        padding = new byte[r+8];
        padding[0] = (byte)0x80;
        for (int i = 1; i <= r-1; i++)
            padding[i] = 0x00;
        for (int i = 0; i < 8; i++)
            padding[r+7-i] = (byte)((count>>>(8*i)) & 0xFFL);
    }

    // Initializes the hash function instance, empties buffers.
    public void hashInit() throws Exception
    {
        state = new long[8];
        intState = new long[8];
        buffer = new byte[64];
        count = 0;
        bufferFilled = 0;
        for (int i = 0; i < 8; i++)
            state[i] = iv[i];
    }

    // Feeds data to the hash function instance, and updates the state as far as possible.
    public void hashUpdate(byte[] data) throws Exception
    {
        int index = bufferFilled;
        int inputLength = data.length;
        int inputOffset = 0;
        int inputRead = 64 - index;
        while (inputLength >= inputRead)
        {
            System.arraycopy(data, inputOffset, buffer, index, inputRead);
            compress();
            count += inputRead << 3;
            index = 0;
            inputLength -= inputRead;
            inputOffset += inputRead;
            inputRead = 64;
        }
        bufferFilled = index + inputLength;
        if (inputLength > 0)
        {
            System.arraycopy(data, inputOffset, buffer, index, inputLength);
            count += inputLength << 3;
        }
    }

    // Finalizes the hash function computation by applying padding (if required) and computing the final hash value.
    public byte[] hashFinal() throws Exception
    {
        pad();
        hashUpdate(padding);
        hashValue = new byte[32];
        byte[] st = stateToBytes(state);
        System.arraycopy(st, 0, hashValue, 0, 32);
        return hashValue;
    }

    // Finalizes the hash function computation by feeding a final blob of data,
    // and computing the final hash value.
    public byte[] hashFinal(byte[] data) throws Exception
    {
        hashUpdate(data);
        return hashFinal();
    }

    // One-pass hashing, byte array to byte array.
    public byte[] hash(byte[] data) throws Exception
    {
        hashInit();
        return hashFinal(data);
    }
    
    // HMAC-SHA256
    public byte[] hMac(byte[] key, byte[] message) throws Exception
    {
        byte[] k0 = new byte[64];
        byte[] k1 = new byte[64];
        byte[] k2 = new byte[64];
        byte[] ih = new byte[32];
        if (key.length > 64)
            System.arraycopy(hash(key), 0, k0, 0, 32);
        else
            System.arraycopy(key, 0, k0, 0, key.length);
        for (int i = 0; i <= 63; i++)
        {
            k1[i] = (byte)(k0[i] ^ 0x5c);
            k2[i] = (byte)(k0[i] ^ 0x36);
        }
        hashInit();
        hashUpdate(k2);
        ih = hashFinal(message);
        hashInit();
        hashUpdate(k1);
        return hashFinal(ih);
    }
}
