package net.deweger.crypto;

import java.security.SecureRandom;
import java.math.BigInteger;

// Utilities for crypto
// Benne de Weger - TU/e - January 2021

public class Util
{
    public SecureRandom secureRandom;

    // constructor
    public Util()
    {
        secureRandom = new SecureRandom();
    }

    // create n random bytes
    public byte[] randomBytes(int n)
    {
        byte[] b = new byte[n];
        secureRandom.nextBytes(b);
        return b;
    }

    // convert one byte to hex string
    public static String byteToHex(byte b)
    {
        StringBuffer buffer = new StringBuffer();
        buffer.append(Character.forDigit((b >> 4) & 0xF, 16));
        buffer.append(Character.forDigit((b & 0xF), 16));
        return buffer.toString();
    }

    // convert byte array to hex string
    public static String bytesToHex(byte[] b)
    {
        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < b.length; i++)
        {
            buffer.append(Character.forDigit((b[i] >> 4) & 0xF, 16));
            buffer.append(Character.forDigit((b[i] & 0xF), 16));
        }
        return buffer.toString();
    }

    // convert hex string to byte array
    public static byte[] hexToBytes(String h) throws Exception
    {
        int n = h.length();
        byte[] b = new byte[n/2];
        for (int i = 0; i < n; i += 2)
        {
            int d1 = Character.digit(h.charAt(i), 16);
            int d2 = Character.digit(h.charAt(i+1), 16);
            if ((d1 == -1) || (d2 == -1))
                throw new Exception("net.deweger.crypto.Util.hexToBytes: illegal character");
            b[i/2] = (byte) ((d1 << 4) + d2);
        }
        return b;
    }

    // convert byte array to hex string with newline after every 32 bytes (64 characters)
    public static String bytesToHex64(byte[] b) throws Exception
    {
        return Util.bytesToHex(b).replaceAll(".{64}", "$0\n");
    }

    // convert hex string to byte array after removing newlines
    public static byte[] hexToBytes64(String h) throws Exception
    {
        return Util.hexToBytes(h.replaceAll("\n",""));
    }

    // convert BigInteger to byte array with leading zero byte removed
    public static byte[] bigIntegerToByteArray(BigInteger bi) throws Exception
    {
        byte[] pb = bi.toByteArray();
        byte[] pb1 = new byte[pb.length];
        if (pb[0] == (byte)0x00)
        {
            pb1 = new byte[pb.length-1];
            System.arraycopy(pb, 1, pb1, 0, pb1.length);
        }
        else
            pb1 = pb;
        return pb1;
    }

    // convert BigInteger to hex string with newline after every 32 bytes (64 characters)
    // and leading zero byte removed
    public static String bigIntegerToHex64(BigInteger bi) throws Exception
    {
        String out = Util.bytesToHex(bigIntegerToByteArray(bi)).replaceAll(".{64}", "$0\n");
        if (out.length() % 65 == 0)
            out = out.substring(0, out.length() - 1);
        return out;
    }

    // compute 3rd root (only for exact third power)
    // uses Newton's method for x^3 - d = 0
    public static BigInteger thirdRoot(BigInteger d) throws Exception
    {
        if (d.compareTo(BigInteger.ONE.negate()) < 0)
            return thirdRoot(d.negate()).negate();
        if (d.compareTo(BigInteger.ONE) <= 0)
            return d;
        BigInteger x = BigInteger.ONE.shiftLeft(d.bitLength()/3);
        BigInteger oldx = BigInteger.ZERO;
        while (!x.equals(oldx))
        {
            oldx = x;
            BigInteger x2 = x.multiply(x);
            x = x.multiply(x2).shiftLeft(1).add(d).divide(x2.multiply(new BigInteger("3")));
            if (x.multiply(x).multiply(x).compareTo(d) < 0)
                throw new Exception("net.deweger.crypto.Util.thirdRoot: input is not an exact third power");
        }
        return x;
    }
}
